﻿# PyU3D first-person camera example.
# Written by MysteriXYZ.

# Declare constants
SCREEN_WIDTH = 800
SCREEN_HEIGHT = 600
SPEED_LIMIT = 200

# Import the PyU3D package
import PyU3D as u3d
from PyU3D._local import *
import PyU3D.host as host

# Psyco is strongly recommended to boost the framerate
import psyco
psyco.full()



# create a custom Camera class, derived from the PyU3D.Camera class
class Camera(u3d.Camera):

  """
  Camera with custom behaviour.

  This camera is aligned to ("follows") an invisible object that is steered
  by the player. As such, this object controls the movement of the camera.
  The alignment procedure provides a way to smooth out the camera motion,
  relative to that of its controller.

  """

  def __init__(self, *args, **kwargs):

    # inherit from PyU3D.Camera
    super(Camera, self).__init__(*args, **kwargs)

    # create the camera movement controller
    self.movement_controller = CamController(
                                              self.x, self.y, 0.,
                                              self.pitch, self.yaw
                                            )

    self.height = 8.



  def step(self):

    """Align camera to controller and update its transformation."""

    self.movement_controller.step()

    smoothing_factor = min(1., 18. * host.getDeltaTime())
    self.smoothness = smoothing_factor

    self.alignTo(
                  self.movement_controller, None,
                  orientation=(True, True, False),
                  smoothness=(smoothing_factor**1.2, smoothing_factor, 0.)
                )

    self.z = self.movement_controller.z + self.height

    self.update()



# create a class to manage transformable camera controllers
class CamController(u3d.Transformable):

  """
  Transformable object that controls the movement of the camera.

  This is an invisible, transformable "dummy" object that the camera can be
  aligned to, thereby enabling smooth camera rotations and movement.

  """

  def __init__(self, *args, **kwargs):

    # inherit from the PyU3D.Transformable class
    super(CamController, self).__init__(*args, **kwargs)

    self.speed = 50. # in distance units per second
    self.mouse_speed = .1


  def step(self):

    mouse_x, mouse_y = mouse.getPosition()

    self.pitch += (mouse_y - SCREEN_HEIGHT / 2) * self.mouse_speed

    if self.pitch < -60.:
      self.pitch = -60.
    if self.pitch > 60.:
      self.pitch = 60.

    self.yaw += (mouse_x - SCREEN_WIDTH / 2) * self.mouse_speed
    self.yaw %= 360.

    mouse.setPosition(SCREEN_WIDTH / 2, SCREEN_HEIGHT / 2)

    # get the number of seconds it currently takes to process a step of the
    # program (i.e. seconds per frame)
    delta_time = host.getDeltaTime()

    # calculate the movement speed in distance units per frame
    speed = self.speed * delta_time

    # Determine the forward and sideways movement vectors, based on the keys
    # that are held down

    advance = (keyb.keyIsDown("up") - keyb.keyIsDown("down"))
    strafe = (keyb.keyIsDown("right") - keyb.keyIsDown("left"))
    advance_vect = DirectionVector(0., self.yaw) * advance
    strafe_vect = DirectionVector(0., self.yaw+90.) * strafe

    # compute the resulting movement vector (which can be normalized so that
    # strafing and advancing simultaneously is done at the same speed as
    # strafing or advancing only)
    move_vect = (advance_vect+strafe_vect)#.normalize()

    # move the camera controller as needed
    if move_vect:
      self.move(move_vect * speed)



# Set U3D options

u3d.setLog("log.txt")
u3d.setZBufferFormat(32)
u3d.setMultiSampleType(0)


# initialize U3D and its host
host.init(
          SCREEN_WIDTH, SCREEN_HEIGHT, SPEED_LIMIT,
          "PyU3D first-person camera example",
          fullscreen=False
         )


# create an object to handle keyboard input
keyb = host.Keyboard()

# create an object to handle the mouse
mouse = host.Mouse()

# hide the cursor
mouse.setVisibility(False)

# should problems arise due to the cursor being hidden, try to use a transparent
# cursor instead of hiding it; comment out the line above and de-comment the
# line below
##mouse.setCursor(None)



# set a background texture
u3d.loadBackgroundTex("Gfx/sea2.jpg")


# Create the scene objects

camera = Camera(SCREEN_WIDTH, SCREEN_HEIGHT, x=130., y=17., pitch=-5., yaw=305.)

ground = u3d.Floor(
                    length=200., width=200.,
                    texture=Texture("Gfx/tiles01.jpg"), u2=50, v2=50
                  )


# set a font
arial12 = Font("Arial", 12, BOLDITALIC)



def main():

  # main loop
  while True:

    # process the current step of the game
    host.step()

    # allow pressing the <Escape> key to end the game
    if keyb.keyIsPressed("esc"):
      host.exit()

    # perform the current step of the camera behaviour
    camera.step()

    # Display important statistics

    arial12.draw(10, 10, "fps: "+str(host.getFPS()))
    arial12.draw(10, 40, "triangles: "+str(getDrawnTriangleCount()))
    arial12.draw(10, 70, "draw calls: "+str(u3d.getDrawCallCount()))


if __name__ == '__main__':
  main()
